/**
 * \file: util_trace.c
 * 
 * \version: $Revision: 1.15 $
 * 
 * \release: $Name: UTIL_r2_E05 $
 *
 * Simplified interface for Trace, which offers among others a printf
 * like function.
 * 
 * \component: Utility library
 * 
 * \author T. Polle / ADIT / SWG / tpolle(o)de.adit-jv.com
 * 
 * \copyright: (c) 2003 - 2008 ADIT Corporation 
 */


#ifdef BUILDENV_LINUX


#else
#include <tk/tkernel.h>
#endif


#include <stdarg.h>
#include <stdio.h>
#include <string.h>

#include "adit_typedef.h"
#include "util_linux_tk_types.h"
#include "util_lib_i.h"

/**
 * Store a signed number with width bytes.
 */
LOCAL void store_d(U8** buf, U32* size, U32 width, S32 d)
{
    S8  b = (S8)d;
    S16 h = (S16)d;
    S32 w = (S32)d;
    if ((*size) >= width)
    {
        switch (width)
        {
        case 1:
            memcpy(*buf, &b, 1);
            (*buf) += 1;
            (*size) -= 1;
            break;
        case 2:
            memcpy(*buf, &h, 2);
            (*buf) += 2;
            (*size) -= 2;
            break;
        default:
            memcpy(*buf, &w, 4);
            (*buf) += 4;
            (*size) -= 4;
            break;
        }
    }
}

/**
 * Store an unsigned number with width bytes.
 */
LOCAL void store_u(U8** buf, U32* size, U32 width, U32 d)
{
    U8 b = (U8)d;
    U16 h = (U16)d;
    U32 w = (U32)d;
    if ((*size) >= width)
    {
        switch (width)
        {
        case 1:
            memcpy(*buf, &b, 1);
            (*buf) += 1;
            (*size) -= 1;
            break;
        case 2:
            memcpy(*buf, &h, 2);
            (*buf) += 2;
            (*size) -= 2;
            break;
        default:
            memcpy(*buf, &w, 4);
            (*buf) += 4;
            (*size) -= 4;
            break;
        }
    }
}

/**
 * Store a string.
 */
LOCAL void store_s(U8** buf, U32* size, U32 width, U8* s)
{
  if (width == 0)
  {
    width = strlen((VP)s) + 1;
  }
  if ((*size) >= width)
  {
    strncpy((VP)(*buf), (VP)s, width);
    (*buf)  += width;
    (*size) -= width;
  }
}

/**
 * Common function for UTIL_tracef and UTIL_atracef.
 * See their definition for the meaning of fmt.
 */
/* PRQA: QAC Message 3408,3625,5013: Preserve compatibility with strings. */
/* PRQA S 3408,3625,5013 L1 */
EXPORT void UTIL_trace_buf(TraceData* data, char* fmt, va_list args)
/* PRQA L:L1 */
{
  U32         length     = 0;
  U8*        buf        = (VP)data->p_ubBuffer;
  U32         size       = data->uwLength;
  TraceData  trace_data ;
  S32        i          = 0;

  memset(&trace_data, 0, sizeof(trace_data));
  trace_data = (*data);

  /* determine the type according to the fmt */
  for (i = 0; fmt[i] != '\0'; i++)
  {
    /* PRQA: QAC Message 3120,3123: Improved readability. */
    /* PRQA S 3120,3123 L1 */
    if (('0' <= fmt[i]) && (fmt[i] <= '9'))
    {
        length = ((10 * length) + ((U32)fmt[i])) - '0';
    }
    /* PRQA L:L1 */
    else
    {
      switch (fmt[i])
      {
      case '%': length = 0;
          break;
	  /*PRQA: Lint Message : usage of va_arg is essential here */
	  /* lint is not able to correctly deduce the argument types and values
	   * that are returned from va_arg
	   */
	  /*lint -save -e530 -e516 -e78 -e64 -e26*/
      case 'd':
          store_d(&buf, &size, length, va_arg(args, S32));
          break;
      case 'u':
          store_u(&buf, &size, length, va_arg(args, U32));
          break;
      case 's':
          store_s(&buf, &size, length, va_arg(args, U8*));
          break;
      default:
          break;
     /*lint -restore */
      }
    }
  }
  trace_data.uwLength -= size;

  /* Any error is simply ignored. */
  UTIL_TraceOut(trace_data.uwLength,
                (U16) trace_data.uhClass,
                (TR_tenTraceLevel) trace_data.ubLevel,
                (U8*)((VP)trace_data.p_ubBuffer));
}

BOOL UTIL_trace_isActive(TraceData* data)
{
  data->fIsActive
      = UTIL_IsClassSelected((U16)data->uhClass,
                             data->ubLevel);
  return data->fIsActive;
}

void UTIL_traceb(TraceData* data, U8* buf, S32 size)
{
  data = data;
  buf  = buf;
  UTIL_TraceOut((U32) size,
                (U16) data->uhClass,
                (TR_tenTraceLevel) data->ubLevel,
                buf);
}

void UTIL_atraceb(TraceData* data, U8* buf, S32 size)
{
  if (0 != data->fIsActive)
  {
    UTIL_traceb(data, buf, size);
  }
}

void UTIL_traceb_mst(TraceStreamData* data, U8* buf, S32 size)
{
  data = data;
  buf  = buf;
  UTIL_TraceBinOutput((U32) data->uwCompId, 
					  (U32) data->uwSockId, 
					  (U32)size,
					  buf);
}

S32 UTIL_trace_chan_reg(LaunchChannel channel)
{
  S32 er = E_OK;
  
	/* Callback function Registration */
	if (channel.p_Callback !=NULL)
  {
    if (FALSE == UTIL_RegisterChannel(channel.enTraceChannel,
                                      (TRACE_CALLBACK)channel.p_Callback))
    {
      er = E_IO;
		}
  }
  return er;
}

S32 UTIL_trace_chan_unreg(LaunchChannel channel)
{
  S32 er = E_OK;
  
	/* Callback function un-Registration */
	if (channel.p_Callback != NULL)
	{
		if (FALSE == UTIL_UnregisterChannel(channel.enTraceChannel,
                                        (TRACE_CALLBACK)channel.p_Callback))
		{
			er = E_IO;
		}
	}
  return er;
}
